<?php

namespace App\Filament\Resources\EventResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class RegistrationsRelationManager extends RelationManager
{
    protected static string $relationship = 'registrations';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('email')
                    ->email()
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('organization')
                    ->maxLength(255),

                Forms\Components\Select::make('attending_as')
                    ->options([
                        'Individual Citizen' => 'Individual Citizen',
                        'Business Representative' => 'Business Representative',
                        'NGO/Civil Society' => 'NGO/Civil Society',
                        'Government Official' => 'Government Official',
                        'Legal Professional' => 'Legal Professional',
                        'Academic/Researcher' => 'Academic/Researcher',
                        'Media' => 'Media',
                    ]),

                Forms\Components\Textarea::make('requirements')
                    ->maxLength(65535),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable(),

                Tables\Columns\TextColumn::make('email')
                    ->searchable(),

                Tables\Columns\TextColumn::make('organization')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make('attending_as')
                    ->badge()
                    ->color(fn(string $state): string => match ($state) {
                        'Government Official' => 'primary',
                        'Legal Professional' => 'success',
                        'Business Representative' => 'warning',
                        default => 'gray',
                    }),

                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('attending_as')
                    ->options([
                        'Individual Citizen' => 'Individual Citizen',
                        'Business Representative' => 'Business Representative',
                        'NGO/Civil Society' => 'NGO/Civil Society',
                        'Government Official' => 'Government Official',
                        'Legal Professional' => 'Legal Professional',
                        'Academic/Researcher' => 'Academic/Researcher',
                        'Media' => 'Media',
                    ]),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
