<?php

namespace App\Filament\Resources;

use App\Filament\Resources\FeedbackResource\Pages;
use App\Models\Feedback;
use Filament\Forms;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;

class FeedbackResource extends Resource
{
    protected static ?string $model = Feedback::class;
    protected static ?string $navigationIcon = 'heroicon-o-chat-bubble-left-ellipsis'; // More appropriate icon
    protected static ?string $navigationGroup = 'Public Engagement';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('User Information')
                    ->columns(2)
                    ->schema([
                        TextInput::make('name')
                            ->required()
                            ->maxLength(255),
                        TextInput::make('email')
                            ->email()
                            ->required()
                            ->maxLength(255),
                    ]),

                Forms\Components\Section::make('Feedback Details')
                    ->schema([
                        TextInput::make('subject')
                            ->required()
                            ->maxLength(255),

                        Select::make('category')
                            ->options([
                                'Existing Law Review' => 'Existing Law Review',
                                'New Law Proposal' => 'New Law Proposal',
                                'Implementation Issues' => 'Implementation Issues',
                                'Other' => 'Other',
                            ])
                            ->required(),

                        Textarea::make('message')
                            ->required()
                            ->columnSpanFull()
                            ->rows(5),
                    ]),

                Forms\Components\Section::make('Administration')
                    ->columns(2)
                    ->schema([
                        TextInput::make('ip_address')
                            ->maxLength(45)
                            ->disabled(), // Disabled since it's auto-generated

                        Forms\Components\Toggle::make('reviewed') // Add the missing field
                            ->label('Mark as Reviewed')
                            ->default(false)
                            ->columnSpan(1),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('created_at')
                    ->dateTime('M j, Y g:i A')
                    ->sortable()
                    ->label('Submitted')
                    ->toggleable(),

                TextColumn::make('name')
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('email')
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('subject')
                    ->searchable()
                    ->limit(30)
                    ->tooltip(fn($record) => $record->subject) // Show full text on hover
                    ->toggleable(),

                TextColumn::make('category')
                    ->badge()
                    ->color(fn(string $state): string => match ($state) {
                        'Existing Law Review' => 'info',
                        'New Law Proposal' => 'primary',
                        'Implementation Issues' => 'warning',
                        'Other' => 'gray',
                    })
                    ->toggleable(),

                IconColumn::make('reviewed')
                    ->boolean()
                    ->sortable()
                    ->label('Reviewed')
                    ->toggleable(),
            ])
            ->filters([
                SelectFilter::make('category')
                    ->options([
                        'Existing Law Review' => 'Existing Law Review',
                        'New Law Proposal' => 'New Law Proposal',
                        'Implementation Issues' => 'Implementation Issues',
                        'Other' => 'Other',
                    ]),

                SelectFilter::make('reviewed')
                    ->options([
                        true => 'Reviewed',
                        false => 'Pending Review',
                    ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\ViewAction::make(), // Add view action
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    // Add bulk mark as reviewed action
                    Tables\Actions\BulkAction::make('markReviewed')
                        ->label('Mark as Reviewed')
                        ->icon('heroicon-o-check')
                        ->action(fn($records) => $records->each->update(['reviewed' => true]))
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->persistSortInSession(); // Remember sorting preferences
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListFeedback::route('/'),
            'create' => Pages\CreateFeedback::route('/create'),
            'edit' => Pages\EditFeedback::route('/{record}/edit'),
        ];
    }
}
