<?php

namespace App\Filament\Resources;

use App\Filament\Resources\MemorandumResource\Pages;
use App\Filament\Resources\MemorandumResource\RelationManagers;
use App\Models\Memorandum;
use Filament\Forms;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class MemorandumResource extends Resource
{
    protected static ?string $model = Memorandum::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Public Engagement';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->required()
                    ->maxLength(255)
                    ->label('Name/Organization'),

                TextInput::make('email')
                    ->email()
                    ->required()
                    ->maxLength(255),

                TextInput::make('title')
                    ->required()
                    ->maxLength(255),

                Select::make('legislation')
                    ->options([
                        'Environmental Protection Law' => 'Environmental Protection Law',
                        'Business Regulation Act' => 'Business Regulation Act',
                        'Tenant Protection Act' => 'Tenant Protection Act',
                        'Digital Rights Bill' => 'Digital Rights Bill',
                        'Other' => 'Other',
                    ])
                    ->required(),

                DateTimePicker::make('deadline')
                    ->required()
                    ->native(false)
                    ->minDate(now())
                    ->label('Submission Deadline'),

                FileUpload::make('document_path')
                    ->label('Memorandum Document')
                    ->directory('memoranda')
                    ->acceptedFileTypes(['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'])
                    ->maxSize(5120)
                    ->required()
                    ->downloadable()
                    ->openable()
                    ->preserveFilenames(),

                Textarea::make('summary')
                    ->required()
                    ->columnSpanFull()
                    ->rows(4),

                TextInput::make('ip_address')
                    ->maxLength(45),
                Forms\Components\Section::make('Administration')
                    ->schema([
                        Forms\Components\Toggle::make('reviewed')
                            ->label('Reviewed')
                            ->default(false),

                        Forms\Components\Toggle::make('acknowledged')
                            ->label('Acknowledged')
                            ->default(false),
                    ])
                    ->columns(2)
                // ->visible(fn() => auth()->user()->admin),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->label('Submitted'),

                TextColumn::make('name')
                    ->searchable(),

                TextColumn::make('email')
                    ->searchable(),

                TextColumn::make('title')
                    ->searchable()
                    ->limit(30),

                TextColumn::make('legislation')
                    ->badge()
                    ->color('primary'),

                IconColumn::make('reviewed')
                    ->boolean()
                    ->sortable()
                    ->label('Reviewed'),

                IconColumn::make('acknowledged')
                    ->boolean()
                    ->sortable()
                    ->label('Acknowledged'),
            ])
            ->filters([
                SelectFilter::make('legislation')
                    ->options([
                        'Environmental Protection Law' => 'Environmental Protection Law',
                        'Business Regulation Act' => 'Business Regulation Act',
                        'Tenant Protection Act' => 'Tenant Protection Act',
                        'Digital Rights Bill' => 'Digital Rights Bill',
                        'Other' => 'Other',
                    ]),
                SelectFilter::make('deadline_passed')
                    ->label('Deadline Status')
                    ->options([
                        'active' => 'Active (Future deadline)',
                        'expired' => 'Expired (Past deadline)',
                    ])
                    ->query(function (Builder $query, array $data) {
                        if ($data['value'] === 'active') {
                            return $query->where('deadline', '>', now());
                        }
                        if ($data['value'] === 'expired') {
                            return $query->where('deadline', '<', now());
                        }
                    }),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
                Tables\Actions\Action::make('download')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->url(fn(Memorandum $record) => storage_path('app/' . $record->document_path))
                    ->openUrlInNewTab(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\BulkAction::make('markReviewed')
                        ->label('Mark as Reviewed')
                        ->icon('heroicon-o-check-circle')
                        ->action(fn($records) => $records->each->update(['reviewed' => true])),
                    Tables\Actions\BulkAction::make('markAcknowledged')
                        ->label('Mark as Acknowledged')
                        ->icon('heroicon-o-document-check')
                        ->action(fn($records) => $records->each->update(['acknowledged' => true])),
                ]),
            ])
            ->defaultSort('deadline', 'asc') // Sort by deadline ascending
            ->persistSortInSession();
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMemoranda::route('/'),
            'create' => Pages\CreateMemorandum::route('/create'),
            'edit' => Pages\EditMemorandum::route('/{record}/edit'),
        ];
    }
}
