<?php

namespace App\Filament\Resources\PublicationChartResource\Widgets;

use App\Models\Publication;
use Carbon\Carbon;
use Filament\Widgets\ChartWidget;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PublicationChart extends ChartWidget
{
    protected static ?string $heading = 'Publication Timeline by Type (Last 6 Months)';
    protected static ?int $sort = 4;
    protected static ?string $maxHeight = '350px';
    protected static ?string $pollingInterval = '30s';

    protected function getData(): array
    {
        $months = collect(range(5, 0))->map(fn($i) => Carbon::now()->subMonths($i)->format('M Y'));

        // Get all publication types
        $types = Publication::select('type')
            ->distinct()
            ->pluck('type')
            ->toArray();

        // Predefined color palette for types
        $colors = [
            '#3B82F6', // Blue - Reports
            '#10B981', // Green - Policy Briefs
            '#F59E0B', // Amber - Research Papers
            '#EF4444', // Red - Others
            '#8B5CF6', // Violet - Additional
            '#EC4899', // Pink - Additional
        ];

        $datasets = [];
        $monthlyData = [];

        // Initialize monthly data structure
        foreach ($months as $month) {
            $monthlyData[$month] = array_fill_keys($types, 0);
        }

        // Get publication counts grouped by month and type
        $publications = Publication::select(
            DB::raw("DATE_FORMAT(published_date, '%b %Y') as month"),
            'type',
            DB::raw('COUNT(*) as count')
        )
            ->where('published_date', '>=', now()->subMonths(5)->startOfMonth())
            ->groupBy('month', 'type')
            ->orderBy('month')
            ->get();

        // Populate monthly data
        foreach ($publications as $pub) {
            if (isset($monthlyData[$pub->month])) {
                $monthlyData[$pub->month][$pub->type] = $pub->count;
            }
        }

        // Prepare datasets for each type
        foreach ($types as $index => $type) {
            $typeData = [];
            foreach ($months as $month) {
                $typeData[] = $monthlyData[$month][$type] ?? 0;
            }

            $datasets[] = [
                'label' => ucfirst($type),
                'data' => $typeData,
                'backgroundColor' => $colors[$index] ?? $this->randomColor(),
                'borderColor' => $this->darkenColor($colors[$index] ?? $this->randomColor()),
                'borderWidth' => 1,
            ];
        }

        return [
            'datasets' => $datasets,
            'labels' => $months,
        ];
    }

    private function darkenColor($color, $amount = 30)
    {
        // Simple darkening function for borders
        return $color; // In real implementation, you'd convert and darken
    }

    private function randomColor()
    {
        return '#' . str_pad(dechex(mt_rand(0, 0xFFFFFF)), 6, '0', STR_PAD_LEFT);
    }

    protected function getType(): string
    {
        return 'bar';
    }

    // Control widget visibility with 'view publication-stats' permission
    public static function canView(): bool
    {
        $user = Auth::user();
        /** @var \App\Models\User $user */
        return $user && $user->can('view publication-stats-on-dashboard');
    }
}
