<?php

namespace App\Filament\Resources\StatsOverviewResource\Widgets;

use App\Models\News;
use App\Models\Publication;
use App\Models\ReformProject;
use App\Models\User;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Illuminate\Support\Facades\Auth;

class StatsOverview extends BaseWidget
{
    protected function getStats(): array
    {
        $stats = [];
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if (!$user) {
            return $stats; // Handle unauthenticated case
        }

        // User Stats Card
        if ($user->can('view user-stats-on-dashboard')) {
            $stats[] = Stat::make('Total Users', User::count())
                ->description('Registered users')
                ->descriptionIcon('heroicon-o-user-group')
                ->color('primary')
                ->chart($this->getMonthlyData(User::class))
                ->extraAttributes(['class' => 'hover:shadow-lg transition-shadow']);
        }

        // Reform Projects Card
        if ($user->can('view reform-stats-on-dashoard')) {
            $stats[] = Stat::make('Total Reform Projects', ReformProject::count())
                ->description('Submitted reforms')
                ->descriptionIcon('heroicon-o-document-text')
                ->color('success')
                ->chart($this->getMonthlyData(ReformProject::class))
                ->extraAttributes(['class' => 'hover:shadow-lg transition-shadow']);
        }

        // Publications Card
        if ($user->can('view publication-stats-on-dashboard')) {
            $stats[] = Stat::make('Publications', Publication::count())
                ->description('Available documents')
                ->descriptionIcon('heroicon-o-document-text')
                ->color('info')
                ->chart($this->getMonthlyData(Publication::class))
                ->extraAttributes(['class' => 'hover:shadow-lg transition-shadow']);
        }

        // News Card
        if ($user->can('view news-stats-on-dashboard')) {
            $stats[] = Stat::make('News Published', News::count())
                ->description('Latest updates')
                ->descriptionIcon('heroicon-o-newspaper')
                ->color('warning')
                ->chart($this->getMonthlyData(News::class))
                ->extraAttributes(['class' => 'hover:shadow-lg transition-shadow']);
        }

        // Active Reform Projects Card
        if ($user->can('view active-reform-stats-on-dashboard')) {
            $stats[] = Stat::make('Active Reform Projects', ReformProject::where('status', 'In Progress')->count())
                ->description('Currently in progress')
                ->descriptionIcon('heroicon-o-arrow-trending-up')
                ->color('info')
                ->chart($this->getStatusData('active'))
                ->extraAttributes(['class' => 'hover:shadow-lg transition-shadow']);
        }

        return $stats;
    }

    private function getMonthlyData(string $model): array
    {
        return $model::query()
            ->selectRaw('MONTH(created_at) as month, COUNT(*) as count')
            ->whereYear('created_at', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->pluck('count')
            ->toArray();
    }

    private function getStatusData(string $status): array
    {
        return ReformProject::query()
            ->selectRaw('MONTH(created_at) as month, COUNT(*) as count')
            ->where('status', $status)
            ->whereYear('created_at', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->pluck('count')
            ->toArray();
    }
}
