<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Casts\AsArrayObject;

class Event extends Model
{
    protected $fillable = [
        'title',
        'description',
        'event_type',
        'start_date',
        'end_date',
        'location',
        'location_details',
        'organizer',
        'organizer_contact',
        'registration_link',
        'agenda',
        'speakers',
        'gallery',
        'supporters',
        'max_attendees',
        'is_published',
        'image_path'
    ];

    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'is_published' => 'boolean',
        'agenda' => 'array',
        'speakers' => 'array',
        'gallery' => 'array',
        'supporters' => 'integer',
        'max_attendees' => 'integer',
    ];

    public function registrations(): HasMany
    {
        return $this->hasMany(EventRegistration::class);
    }

    /**
     * Get the featured image URL
     */
    public function getFeaturedImageUrlAttribute()
    {
        return $this->image_path ? asset('storage/' . $this->image_path) : null;
    }

    /**
     * Get gallery image URLs
     */
    public function getGalleryUrlsAttribute()
    {
        if (!$this->gallery) return [];

        return array_map(function ($path) {
            return asset('storage/' . $path);
        }, $this->gallery);
    }

    /**
     * Check if event has passed
     */
    public function getIsPastEventAttribute()
    {
        return now()->gt($this->end_date);
    }

    /**
     * Check if event is upcoming
     */
    public function getIsUpcomingEventAttribute()
    {
        return now()->lt($this->start_date);
    }

    /**
     * Check if event is happening now
     */
    public function getIsCurrentEventAttribute()
    {
        return now()->between($this->start_date, $this->end_date);
    }

    /**
     * Get formatted date range
     */
    public function getFormattedDateRangeAttribute()
    {
        if ($this->start_date->format('Y-m-d') === $this->end_date->format('Y-m-d')) {
            return $this->start_date->format('M d, Y') . ' from ' .
                $this->start_date->format('g:i A') . ' to ' .
                $this->end_date->format('g:i A');
        }

        return $this->start_date->format('M d, g:i A') . ' - ' .
            $this->end_date->format('M d, Y g:i A');
    }

    /**
     * Scope for published events
     */
    public function scopePublished($query)
    {
        return $query->where('is_published', true);
    }

    /**
     * Scope for upcoming events
     */
    public function scopeUpcoming($query)
    {
        return $query->where('end_date', '>', now());
    }

    /**
     * Scope for past events
     */
    public function scopePast($query)
    {
        return $query->where('end_date', '<', now());
    }

    /**
     * Scope for current events
     */
    public function scopeCurrent($query)
    {
        return $query->where('start_date', '<=', now())
            ->where('end_date', '>=', now());
    }
}
